import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { UserEntity } from './user.entity';

@Injectable()
export class UsersService {
  constructor(
    @InjectRepository(UserEntity)
    private readonly usersRepo: Repository<UserEntity>
  ) {}

  findByEmail(email: string) {
    return this.usersRepo.findOne({ 
      where: { email }, 
      select: [
        'id', 
        'email', 
        'name', 
        'userType', 
        'passwordHash', 
        'isEmailVerified',
        'emailVerificationToken',
        'emailVerificationExpires',
        'passwordResetToken',
        'passwordResetExpires'
      ] 
    });
  }

  findById(id: string) {
    return this.usersRepo.findOne({ where: { id } });
  }

  async createUser(params: { 
    email: string; 
    name?: string; 
    passwordHash: string; 
    userType?: 'professional' | 'company';
    isEmailVerified?: boolean;
    emailVerificationToken?: string;
    emailVerificationExpires?: Date;
  }) {
    const user = this.usersRepo.create({
      email: params.email,
      name: params.name || params.email.split('@')[0],
      passwordHash: params.passwordHash,
      userType: params.userType || 'professional',
      isEmailVerified: params.isEmailVerified || false,
      emailVerificationToken: params.emailVerificationToken,
      emailVerificationExpires: params.emailVerificationExpires
    });
    return this.usersRepo.save(user);
  }

  async updateUser(id: string, updates: Partial<UserEntity>) {
    await this.usersRepo.update(id, updates);
    return this.findById(id);
  }

  async searchByName(searchTerm: string) {
    return this.usersRepo
      .createQueryBuilder('user')
      .where('LOWER(user.name) LIKE LOWER(:searchTerm)', { searchTerm: `%${searchTerm}%` })
      .select(['user.id', 'user.name', 'user.email', 'user.userType', 'user.avatarUrl'])
      .limit(20)
      .getMany();
  }

  async searchCompanies(searchTerm: string) {
    return this.usersRepo
      .createQueryBuilder('user')
      .where('user.userType = :userType', { userType: 'company' })
      .andWhere('LOWER(user.name) LIKE LOWER(:searchTerm)', { searchTerm: `%${searchTerm}%` })
      .select(['user.id', 'user.name', 'user.email', 'user.userType', 'user.avatarUrl'])
      .limit(10)
      .getMany();
  }
}


