import "reflect-metadata";
import { NestFactory } from "@nestjs/core";
import { AppModule } from "./app.module";
import { DataSource } from "typeorm";
import { UserEntity } from "./users/user.entity";
import { ProfileEntity } from "./profile/profile.entity";
import { PostEntity } from "./posts/post.entity";
import { LikesEntity } from "./posts/likes.entity";
import { JobEntity } from "./jobs/job.entity";
import { NewsEntity } from "./news/news.entity";
import { VesselEntity } from "./vessels/vessel.entity";
import { ConnectionEntity } from "./network/connection.entity";
import { MessageEntity } from "./messages/message.entity";
import * as bcrypt from "bcryptjs";
import { existsSync, mkdirSync } from "fs";

async function ensureUploadDirs() {
  const dirs = ["uploads", "uploads/avatars", "uploads/news", "uploads/posts"];
  dirs.forEach((d) => {
    if (!existsSync(d)) mkdirSync(d);
  });
}

async function bootstrap() {
  await ensureUploadDirs();
  const app = await NestFactory.createApplicationContext(AppModule);
  const dataSource = app.get(DataSource);

  const userRepo = dataSource.getRepository(UserEntity);
  const profileRepo = dataSource.getRepository(ProfileEntity);
  const postRepo = dataSource.getRepository(PostEntity);
  const jobRepo = dataSource.getRepository(JobEntity);
  const newsRepo = dataSource.getRepository(NewsEntity);
  const vesselRepo = dataSource.getRepository(VesselEntity);
  const connectionRepo = dataSource.getRepository(ConnectionEntity);
  const messageRepo = dataSource.getRepository(MessageEntity);
  const likesRepo = dataSource.getRepository(LikesEntity);

  // Clear existing
  await messageRepo.delete({});
  await connectionRepo.delete({});
  await postRepo.delete({});
  await newsRepo.delete({});
  await jobRepo.delete({});
  await vesselRepo.delete({});
  await profileRepo.delete({});
  await userRepo.delete({});

  // Users
  const users = [
    {
      email: "captain@maritime.com",
      name: "Captain John",
      role: "Captain",
      password: "Captain123!",
    },
    {
      email: "engineer@shipping.com",
      name: "Chief Engineer",
      role: "Engineer",
      password: "Engineer456!",
    },
    {
      email: "port@manager.com",
      name: "Port Manager",
      role: "Manager",
      password: "Port789!",
    },
  ];

  const createdUsers: UserEntity[] = [];
  for (const u of users) {
    const passwordHash = await bcrypt.hash(u.password, 10);
    const user = userRepo.create({
      email: u.email,
      name: u.name,
      userType: 'professional',
      passwordHash,
    });
    createdUsers.push(await userRepo.save(user));
  }

  // Profiles
  for (const u of createdUsers) {
    const profile = profileRepo.create({
      user: u,
      currentCompany: u.userType === 'professional' ? "Maritime Shipping Co." : undefined,
      experience: "10 years",
      certifications: ["STCW"],
      skills: ["Navigation", "Safety Management"],
    });
    await profileRepo.save(profile);
  }

  // Posts
  for (const u of createdUsers) {
    // Create the post
    const post = postRepo.create({
      author: u,
      content: `Hello from ${u.name}!`,
    });
    const savedPost = await postRepo.save(post);

    // Create random likes for this post
    const randomLikeCount = Math.floor(Math.random() * 20);
    const randomUsers = createdUsers
      .sort(() => 0.5 - Math.random())
      .slice(0, randomLikeCount);

    const likes = randomUsers.map((user) =>
      likesRepo.create({
        user: user,
        post: savedPost,
      })
    );

    await likesRepo.save(likes);
  }

  // Jobs
  const job1 = jobRepo.create({
    title: "Chief Engineer",
    description: "Experienced chief engineer needed for maritime operations",
    location: "Singapore",
    salaryRange: "$8,000 - $12,000",
    jobType: "full-time",
    requirements: "10+ years experience, STCW certification",
    company: createdUsers[0],
  });
  await jobRepo.save(job1);

  const job2 = jobRepo.create({
    title: "Vessel Captain",
    description: "Seeking experienced vessel captain",
    location: "Rotterdam",
    salaryRange: "$10,000 - $15,000",
    jobType: "full-time",
    requirements: "Master's license, 15+ years experience",
    company: createdUsers[0],
  });
  await jobRepo.save(job2);

  // News
  await newsRepo.save(
    newsRepo.create({
      title: "New Maritime Regulations Coming",
      summary: "Important updates to international shipping regulations...",
      author: "Maritime News",
    })
  );

  // Vessels
  await vesselRepo.save(
    vesselRepo.create({
      name: "MV Ocean Star",
      type: "Container Ship",
      status: "Available",
      capacity: "5000 TEU",
      location: "Singapore Port",
    })
  );

  // Connections
  await connectionRepo.save(
    connectionRepo.create({
      user: createdUsers[0],
      connection: createdUsers[1],
      status: "accepted",
    })
  );

  // Messages
  const message = messageRepo.create({
    sender: createdUsers[1],
    recipient: createdUsers[0],
    content: "Hello! How are you?",
  });
  await messageRepo.save(message);

  // eslint-disable-next-line no-console
  console.log("Seed completed.");
  await app.close();
}

bootstrap().catch((e) => {
  // eslint-disable-next-line no-console
  console.error(e);
  process.exit(1);
});
