import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { ProfileEntity } from './profile.entity';
import { UserEntity } from '../users/user.entity';
import { ConnectionEntity } from '../network/connection.entity';
import { FollowerEntity } from '../network/follower.entity';
import { PostEntity } from '../posts/post.entity';

@Injectable()
export class ProfileService {
  constructor(
    @InjectRepository(ProfileEntity)
    private readonly profiles: Repository<ProfileEntity>,
    @InjectRepository(UserEntity)
    private readonly users: Repository<UserEntity>,
    @InjectRepository(ConnectionEntity)
    private readonly connections: Repository<ConnectionEntity>,
    @InjectRepository(FollowerEntity)
    private readonly followers: Repository<FollowerEntity>,
    @InjectRepository(PostEntity)
    private readonly posts: Repository<PostEntity>
  ) {}

  findByUserId(userId: string) {
    return this.profiles.findOne({ where: { user: { id: userId } } });
  }

  async upsert(userId: string, data: Partial<ProfileEntity>) {
    const user = await this.users.findOneOrFail({ where: { id: userId } });
    let profile = await this.findByUserId(userId);
    if (!profile) {
      profile = this.profiles.create({ user, ...data });
    } else {
      Object.assign(profile, data);
    }
    return this.profiles.save(profile);
  }

  async getProfileStats(userId: string) {
    const [connectionsCount, followersCount, followingCount, postsCount] = await Promise.all([
      this.connections.count({
        where: [
          { user: { id: userId }, status: 'accepted' },
          { connection: { id: userId }, status: 'accepted' }
        ]
      }),
      this.followers.count({
        where: { following: { id: userId } }
      }),
      this.followers.count({
        where: { follower: { id: userId } }
      }),
      this.posts.count({
        where: { author: { id: userId } }
      })
    ]);

    return {
      connections: connectionsCount,
      followers: followersCount,
      following: followingCount,
      posts: postsCount
    };
  }

  async getProfileWithStats(userId: string) {
    const user = await this.users.findOne({ where: { id: userId } });
    const profile = await this.findByUserId(userId);
    const stats = await this.getProfileStats(userId);
    
    // Merge user and profile data
    return {
      id: user?.id,
      name: user?.name,
      email: user?.email,
      userType: user?.userType,
      avatarUrl: user?.avatarUrl,
      phone: user?.phone,
      ...profile,
      stats
    };
  }

  async updateAvatar(userId: string, filename: string) {
    const user = await this.users.findOne({ where: { id: userId } });
    if (!user) {
      throw new Error('User not found');
    }
    
    user.avatarUrl = `/uploads/avatars/${filename}`;
    await this.users.save(user);
    
    return this.getProfileWithStats(userId);
  }
}


