import {
  Body,
  Controller,
  Delete,
  Get,
  Param,
  Patch,
  Post,
  UploadedFile,
  UseGuards,
  UseInterceptors,
  Req,
} from "@nestjs/common";
import { JwtAuthGuard } from "../auth/jwt-auth.guard";
import { PostsService } from "./posts.service";
import { FileInterceptor } from "@nestjs/platform-express";
import { diskStorage } from "multer";
import { extname, join } from "path";
import * as fs from "fs";
import type { Express, Request } from "express";

interface JwtRequest extends Request {
  user: {
    userId: string;
    email: string;
    role: string;
  };
}

type FilenameCallback = (error: Error | null, filename: string) => void;

export function storageFactory(folder: string) {
  const uploadPath = join(process.cwd(), "uploads", folder);
  if (!fs.existsSync(uploadPath)) fs.mkdirSync(uploadPath, { recursive: true });

  return diskStorage({
    destination: uploadPath,
    filename: (
      _req: Express.Request,
      file: Express.Multer.File,
      cb: FilenameCallback
    ) => {
      const uniqueSuffix = `${Date.now()}-${Math.round(Math.random() * 1e9)}`;
      const ext = extname(file.originalname);
      const filename = `${uniqueSuffix}${ext}`;
      cb(null, filename);
    },
  });
}

@Controller("posts")
@UseGuards(JwtAuthGuard)
export class PostsController {
  constructor(private readonly postsService: PostsService) {}

  // 🔹 Get all posts
  @Get()
  list() {
    return this.postsService.findAll();
  }

  // 🔹 Get single post
  @Get(":id")
  get(@Param("id") id: string) {
    return this.postsService.findOne(id);
  }

  // 🔹 Create a post (with optional image)
  @Post()
  @UseInterceptors(
    FileInterceptor("image", {
      storage: storageFactory("posts"),
    })
  )
  async create(
    @Body() body: any,
    @UploadedFile() image: Express.Multer.File | undefined,
    @Req() req: JwtRequest
  ) {
    let imageUrl: string | undefined;
    if (image) {
      const savedFilename = image.filename || image.originalname;
      const savedPath = join(process.cwd(), "uploads", "posts", savedFilename);
      if (fs.existsSync(savedPath))
        imageUrl = `/uploads/posts/${savedFilename}`;
    }

    // Use authenticated user ID from JWT token
    const authorId = req.user.userId;

    const postData: any = {
      content: body.content,
      author: { id: authorId },
    };

    const post = await this.postsService.createWithImage(postData, imageUrl);
    
    if (!post) {
      throw new Error('Failed to create post');
    }
    
    return {
      id: post.id,
      content: post.content,
      imageUrl: post.imageUrl,
      createdAt: post.createdAt,
      updatedAt: post.updatedAt,
      author: post.author,
    };
  }

  // 🔹 Update a post (with optional image)
  @Patch(":id")
  @UseInterceptors(
    FileInterceptor("image", {
      storage: storageFactory("posts"),
    })
  )
  async update(
    @Param("id") id: string,
    @Body() body: any,
    @UploadedFile() image: Express.Multer.File | undefined,
    @Req() req: JwtRequest
  ) {
    // Verify authorization - user can only edit their own posts
    await this.postsService.verifyPostOwnership(id, req.user.userId);

    let imageUrl: string | undefined;
    if (image) {
      const savedFilename = image.filename || image.originalname;
      const savedPath = join(process.cwd(), "uploads", "posts", savedFilename);
      if (fs.existsSync(savedPath))
        imageUrl = `/uploads/posts/${savedFilename}`;
    }

    const updateData: any = {
      content: body.content,
    };

    // Handle image update or removal
    if (imageUrl) {
      updateData.imageUrl = imageUrl;
    } else if (body.removeImage === 'true' || body.removeImage === true) {
      updateData.imageUrl = null;
    }

    const updatedPost = await this.postsService.update(id, updateData, req.user.userId);
    
    // Return the updated post with all relations
    return this.postsService.findOne(id);
  }

  // 🔹 Delete a post
  @Delete(":id")
  remove(@Param("id") id: string) {
    return this.postsService.remove(id);
  }

  // ==============================
  // ❤️ LIKE ENDPOINTS
  // ==============================

  @Get(":id/likes")
  async getLikes(@Param("id") postId: string) {
    return this.postsService.getLikesByPost(postId);
  }

  // 🔹 Like a post
  @Post(":id/like")
  async likePost(@Param("id") postId: string, @Req() req: JwtRequest) {
    return this.postsService.addLike(postId, req.user.userId);
  }

  // 🔹 Unlike a post
  @Delete(":id/unlike")
  async unlikePost(@Param("id") postId: string, @Req() req: JwtRequest) {
    return this.postsService.removeLike(postId,  req.user.userId);
  }

  // ==============================
  // 💬 COMMENT ENDPOINTS
  // ==============================

  // 🔹 Add a comment
  @Post(":id/comments")
  async addComment(
    @Param("id") postId: string,
    @Body() body: any,
    @Req() req: Request
  ) {
    const user = req.user as any;
    return this.postsService.addComment(postId, user.id, body.content);
  }

  // 🔹 Edit a comment
  @Patch("comments/:commentId")
  async updateComment(
    @Param("commentId") commentId: string,
    @Body() body: any,
    @Req() req: Request
  ) {
    const user = req.user as any;
    return this.postsService.updateComment(commentId, user.id, body.content);
  }

  // 🔹 Delete a comment
  @Delete("comments/:commentId")
  async deleteComment(
    @Param("commentId") commentId: string,
    @Req() req: Request
  ) {
    const user = req.user as any;
    return this.postsService.deleteComment(commentId, user.id);
  }

  // 🔹 Get all comments for a post
  @Get(":id/comments")
  async getComments(@Param("id") postId: string) {
    return this.postsService.getCommentsByPost(postId);
  }
}
