import { Injectable, NotFoundException, BadRequestException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { ConnectionEntity } from './connection.entity';
import { FollowerEntity } from './follower.entity';
import { UserEntity } from '../users/user.entity';

@Injectable()
export class NetworkService {
  constructor(
    @InjectRepository(ConnectionEntity)
    private readonly connectionRepo: Repository<ConnectionEntity>,
    @InjectRepository(FollowerEntity)
    private readonly followerRepo: Repository<FollowerEntity>
  ) {}

  // Connection methods
  async getConnections(userId: string) {
    const connections = await this.connectionRepo.find({
      where: [
        { user: { id: userId }, status: 'accepted' },
        { connection: { id: userId }, status: 'accepted' }
      ],
      relations: ['user', 'user.profile', 'connection', 'connection.profile']
    });
    
    // Transform to return the other user in the connection
    return connections.map(conn => {
      const otherUser = conn.user.id === userId ? conn.connection : conn.user;
      return {
        id: otherUser.id,
        name: otherUser.name,
        email: otherUser.email,
        userType: otherUser.userType,
        avatarUrl: otherUser.avatarUrl,
        currentRole: otherUser.profile?.currentRole,
        companyName: otherUser.profile?.companyName,
        connectionId: conn.id
      };
    });
  }

  async getSentRequests(userId: string) {
    const requests = await this.connectionRepo.find({
      where: { user: { id: userId }, status: 'pending' },
      relations: ['connection', 'connection.profile']
    });
    
    // Transform to include recipient for frontend compatibility
    return requests.map(req => ({
      ...req,
      recipient: req.connection,
      recipientId: req.connection.id
    }));
  }

  async getReceivedRequests(userId: string) {
    const requests = await this.connectionRepo.find({
      where: { connection: { id: userId }, status: 'pending' },
      relations: ['user', 'user.profile']
    });
    
    // Transform to include sender for frontend compatibility
    return requests.map(req => ({
      ...req,
      sender: req.user,
      senderId: req.user.id
    }));
  }

  async sendConnectionRequest(userId: string, targetUserId: string) {
    if (userId === targetUserId) {
      throw new BadRequestException('Cannot connect with yourself');
    }

    const existing = await this.connectionRepo.findOne({
      where: [
        { user: { id: userId }, connection: { id: targetUserId } },
        { user: { id: targetUserId }, connection: { id: userId } }
      ]
    });

    if (existing) {
      throw new BadRequestException('Connection request already exists');
    }

    const connection = this.connectionRepo.create({
      user: { id: userId } as UserEntity,
      connection: { id: targetUserId } as UserEntity,
      status: 'pending'
    });

    const saved = await this.connectionRepo.save(connection);
    
    // Load the full connection with relations for the response
    const fullConnection = await this.connectionRepo.findOne({
      where: { id: saved.id },
      relations: ['user', 'user.profile', 'connection', 'connection.profile']
    });
    
    if (!fullConnection) {
      throw new NotFoundException('Failed to create connection request');
    }
    
    // Return with recipient for frontend compatibility
    return {
      ...fullConnection,
      recipient: fullConnection.connection,
      recipientId: fullConnection.connection.id
    };
  }

  async acceptConnection(requestId: string, userId: string) {
    const request = await this.connectionRepo.findOne({
      where: { id: requestId },
      relations: ['user', 'user.profile', 'connection', 'connection.profile']
    });

    if (!request) {
      throw new NotFoundException('Connection request not found');
    }

    if (request.connection.id !== userId) {
      throw new BadRequestException('Not authorized to accept this request');
    }

    request.status = 'accepted';
    const saved = await this.connectionRepo.save(request);
    
    // Return the connection data for the frontend (the user who sent the request)
    return {
      id: saved.user.id,
      name: saved.user.name,
      email: saved.user.email,
      userType: saved.user.userType,
      avatarUrl: saved.user.avatarUrl,
      currentRole: saved.user.profile?.currentRole,
      companyName: saved.user.profile?.companyName
    };
  }

  async rejectConnection(requestId: string, userId: string) {
    const request = await this.connectionRepo.findOne({
      where: { id: requestId }
    });

    if (!request) {
      throw new NotFoundException('Connection request not found');
    }

    if (request.connection.id !== userId) {
      throw new BadRequestException('Not authorized to reject this request');
    }

    return this.connectionRepo.delete(requestId);
  }

  async removeConnection(connectionId: string) {
    const connection = await this.connectionRepo.findOne({
      where: { id: connectionId }
    });

    if (!connection) {
      throw new NotFoundException('Connection not found');
    }

    return this.connectionRepo.delete(connectionId);
  }

  // Follower methods
  async followUser(followerId: string, followingId: string) {
    if (followerId === followingId) {
      throw new BadRequestException('Cannot follow yourself');
    }

    const existing = await this.followerRepo.findOne({
      where: {
        follower: { id: followerId },
        following: { id: followingId }
      }
    });

    if (existing) {
      throw new BadRequestException('Already following this user');
    }

    const follow = this.followerRepo.create({
      follower: { id: followerId } as UserEntity,
      following: { id: followingId } as UserEntity
    });

    return this.followerRepo.save(follow);
  }

  async unfollowUser(followerId: string, followingId: string) {
    const follow = await this.followerRepo.findOne({
      where: {
        follower: { id: followerId },
        following: { id: followingId }
      }
    });

    if (!follow) {
      throw new NotFoundException('Follow relationship not found');
    }

    return this.followerRepo.delete(follow.id);
  }

  async getFollowers(userId: string) {
    const followers = await this.followerRepo.find({
      where: { following: { id: userId } },
      relations: ['follower', 'follower.profile']
    });
    
    // Transform to return follower user data
    return followers.map(follow => ({
      id: follow.follower.id,
      name: follow.follower.name,
      email: follow.follower.email,
      userType: follow.follower.userType,
      avatarUrl: follow.follower.avatarUrl,
      currentRole: follow.follower.profile?.currentRole,
      companyName: follow.follower.profile?.companyName
    }));
  }

  async getFollowing(userId: string) {
    const following = await this.followerRepo.find({
      where: { follower: { id: userId } },
      relations: ['following', 'following.profile']
    });
    
    // Transform to return following user data
    return following.map(follow => ({
      id: follow.following.id,
      name: follow.following.name,
      email: follow.following.email,
      userType: follow.following.userType,
      avatarUrl: follow.following.avatarUrl,
      currentRole: follow.following.profile?.currentRole,
      companyName: follow.following.profile?.companyName
    }));
  }

  async getFollowerCount(userId: string): Promise<number> {
    return this.followerRepo.count({
      where: { following: { id: userId } }
    });
  }

  async getFollowingCount(userId: string): Promise<number> {
    return this.followerRepo.count({
      where: { follower: { id: userId } }
    });
  }
}


