import { Body, Controller, Delete, Get, Param, Patch, Post, Req, UseGuards } from '@nestjs/common';
import { JwtAuthGuard } from '../auth/jwt-auth.guard';
import { NetworkService } from './network.service';

@Controller('network')
@UseGuards(JwtAuthGuard)
export class NetworkController {
  constructor(private readonly networkService: NetworkService) {}

  // Connection endpoints
  @Get('connections')
  getConnections(@Req() req: any) {
    return this.networkService.getConnections(req.user.userId);
  }

  @Get('requests/sent')
  getSentRequests(@Req() req: any) {
    return this.networkService.getSentRequests(req.user.userId);
  }

  @Get('requests/received')
  getReceivedRequests(@Req() req: any) {
    return this.networkService.getReceivedRequests(req.user.userId);
  }

  @Post('connect/:userId')
  sendConnectionRequest(@Req() req: any, @Param('userId') targetUserId: string) {
    return this.networkService.sendConnectionRequest(req.user.userId, targetUserId);
  }

  @Patch('requests/:id/accept')
  acceptConnection(@Req() req: any, @Param('id') requestId: string) {
    return this.networkService.acceptConnection(requestId, req.user.userId);
  }

  @Patch('requests/:id/reject')
  rejectConnection(@Req() req: any, @Param('id') requestId: string) {
    return this.networkService.rejectConnection(requestId, req.user.userId);
  }

  @Delete('connections/:id')
  removeConnection(@Param('id') connectionId: string) {
    return this.networkService.removeConnection(connectionId);
  }

  // Follower endpoints
  @Post('follow/:userId')
  followUser(@Req() req: any, @Param('userId') targetUserId: string) {
    return this.networkService.followUser(req.user.userId, targetUserId);
  }

  @Delete('follow/:userId')
  unfollowUser(@Req() req: any, @Param('userId') targetUserId: string) {
    return this.networkService.unfollowUser(req.user.userId, targetUserId);
  }

  @Get('followers/:userId')
  getFollowers(@Param('userId') userId: string) {
    return this.networkService.getFollowers(userId);
  }

  @Get('following/:userId')
  getFollowing(@Param('userId') userId: string) {
    return this.networkService.getFollowing(userId);
  }
}


