import { Test, TestingModule } from '@nestjs/testing';
import { getRepositoryToken } from '@nestjs/typeorm';
import { NetworkService } from './network.service';
import { ConnectionEntity } from './connection.entity';
import { FollowerEntity } from './follower.entity';
import * as fc from 'fast-check';

/**
 * Feature: maritime-app-enhancements
 * Property 15: Connection acceptance creates bidirectional relationship
 * Property 16: Connection rejection removes request
 * Property 20: Connection count symmetry
 * Validates: Requirements 6.3, 6.4, 7.3
 */

describe('NetworkService Connection Property Tests', () => {
  let service: NetworkService;
  let connectionRepo: any;

  beforeEach(async () => {
    const mockConnectionRepo = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn((data: any) => data),
      save: jest.fn((data: any) => Promise.resolve({ id: 'conn-id', ...data })),
      delete: jest.fn()
    };

    const mockFollowerRepo = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn(),
      save: jest.fn(),
      delete: jest.fn(),
      count: jest.fn()
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        NetworkService,
        {
          provide: getRepositoryToken(ConnectionEntity),
          useValue: mockConnectionRepo
        },
        {
          provide: getRepositoryToken(FollowerEntity),
          useValue: mockFollowerRepo
        }
      ]
    }).compile();

    service = module.get<NetworkService>(NetworkService);
    connectionRepo = module.get(getRepositoryToken(ConnectionEntity));
  });

  describe('Property 15: Connection acceptance creates bidirectional relationship', () => {
    it('should create accepted connection for both users', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.uuid(),
          fc.uuid(),
          async (userId1, userId2) => {
            fc.pre(userId1 !== userId2);

            const mockRequest = {
              id: 'request-id',
              user: { 
                id: userId1, 
                name: 'User 1', 
                email: 'user1@test.com', 
                userType: 'professional',
                profile: { currentRole: 'Engineer' }
              },
              connection: { id: userId2 },
              status: 'pending'
            };

            connectionRepo.findOne.mockResolvedValue(mockRequest);
            connectionRepo.save.mockResolvedValue({ ...mockRequest, status: 'accepted' });

            const result = await service.acceptConnection('request-id', userId2);

            expect(result.id).toBe(userId1);
            expect(connectionRepo.save).toHaveBeenCalled();
          }
        ),
        { numRuns: 50 }
      );
    });
  });

  describe('Property 16: Connection rejection removes request', () => {
    it('should remove connection request when rejected', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.uuid(),
          fc.uuid(),
          async (userId1, userId2) => {
            fc.pre(userId1 !== userId2);

            const mockRequest = {
              id: 'request-id',
              user: { id: userId1 },
              connection: { id: userId2 },
              status: 'pending'
            };

            connectionRepo.findOne.mockResolvedValue(mockRequest);
            connectionRepo.delete.mockResolvedValue({ affected: 1 });

            await service.rejectConnection('request-id', userId2);

            expect(connectionRepo.delete).toHaveBeenCalledWith('request-id');
          }
        ),
        { numRuns: 50 }
      );
    });
  });

  describe('Property 20: Connection count symmetry', () => {
    it('should show symmetric connection counts after acceptance', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.uuid(),
          fc.uuid(),
          fc.nat(10),
          async (userId1, userId2, initialCount) => {
            fc.pre(userId1 !== userId2);

            // Mock initial state
            const initialConnections = Array(initialCount).fill(null).map((_, i) => ({
              id: `conn-${i}`,
              user: { id: `user-${i}`, name: `User ${i}`, email: `user${i}@test.com`, userType: 'professional' },
              connection: { id: userId1, name: 'Current User', email: 'current@test.com', userType: 'professional' },
              status: 'accepted'
            }));
            
            const newConnection = {
              id: 'new-conn',
              user: { id: userId2, name: 'New User', email: 'new@test.com', userType: 'professional' },
              connection: { id: userId1, name: 'Current User', email: 'current@test.com', userType: 'professional' },
              status: 'accepted'
            };
            
            connectionRepo.findOne
              .mockResolvedValueOnce(null) // First call for existing connection check
              .mockResolvedValueOnce(newConnection); // Second call for loading full connection
            connectionRepo.find
              .mockResolvedValueOnce(initialConnections)
              .mockResolvedValueOnce([...initialConnections, newConnection]);

            const countBefore = (await service.getConnections(userId1)).length;
            await service.sendConnectionRequest(userId1, userId2);
            const countAfter = (await service.getConnections(userId1)).length;

            // After accepting a connection, count should increase by 1
            expect(countAfter).toBe(countBefore + 1);
          }
        ),
        { numRuns: 50 }
      );
    });
  });
});
