import { Injectable, BadRequestException, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { MessageEntity } from './message.entity';
import { UserEntity } from '../users/user.entity';
import { ConnectionEntity } from '../network/connection.entity';

@Injectable()
export class MessagesService {
  constructor(
    @InjectRepository(MessageEntity)
    private readonly messageRepo: Repository<MessageEntity>,
    @InjectRepository(ConnectionEntity)
    private readonly connectionRepo: Repository<ConnectionEntity>
  ) {}

  async checkConnection(userId: string, otherUserId: string): Promise<boolean> {
    const connection = await this.connectionRepo.findOne({
      where: [
        { user: { id: userId }, connection: { id: otherUserId }, status: 'accepted' },
        { user: { id: otherUserId }, connection: { id: userId }, status: 'accepted' }
      ]
    });
    return !!connection;
  }

  async sendMessage(senderId: string, recipientId: string, content: string) {
    // Check if users are connected
    const areConnected = await this.checkConnection(senderId, recipientId);
    if (!areConnected) {
      throw new BadRequestException('You can only message users you are connected with');
    }

    const message = this.messageRepo.create({
      sender: { id: senderId } as UserEntity,
      recipient: { id: recipientId } as UserEntity,
      content,
      isRead: false
    });

    return this.messageRepo.save(message);
  }

  async getConversation(userId: string, otherUserId: string) {
    // Check if users are connected
    const areConnected = await this.checkConnection(userId, otherUserId);
    if (!areConnected) {
      throw new BadRequestException('You can only view conversations with connected users');
    }

    const messages = await this.messageRepo.find({
      where: [
        { sender: { id: userId }, recipient: { id: otherUserId } },
        { sender: { id: otherUserId }, recipient: { id: userId } }
      ],
      order: { createdAt: 'ASC' }
    });

    return messages;
  }

  async getConversations(userId: string) {
    // Get all messages where user is sender or recipient
    const messages = await this.messageRepo
      .createQueryBuilder('message')
      .leftJoinAndSelect('message.sender', 'sender')
      .leftJoinAndSelect('message.recipient', 'recipient')
      .where('message.senderId = :userId OR message.recipientId = :userId', { userId })
      .orderBy('message.createdAt', 'DESC')
      .getMany();

    // Group by conversation partner
    const conversationsMap = new Map();
    
    for (const message of messages) {
      const partnerId = message.sender.id === userId ? message.recipient.id : message.sender.id;
      
      if (!conversationsMap.has(partnerId)) {
        const partner = message.sender.id === userId ? message.recipient : message.sender;
        conversationsMap.set(partnerId, {
          user: {
            id: partner.id,
            name: partner.name,
            avatarUrl: partner.avatarUrl,
            userType: partner.userType
          },
          lastMessage: message,
          unreadCount: 0
        });
      }
      
      // Count unread messages
      if (message.recipient.id === userId && !message.isRead) {
        conversationsMap.get(partnerId).unreadCount++;
      }
    }

    return Array.from(conversationsMap.values());
  }

  async markAsRead(messageId: string, userId: string) {
    const message = await this.messageRepo.findOne({ where: { id: messageId } });
    
    if (!message) {
      throw new NotFoundException('Message not found');
    }
    
    if (message.recipient.id !== userId) {
      throw new BadRequestException('You can only mark your own messages as read');
    }

    message.isRead = true;
    return this.messageRepo.save(message);
  }

  async markConversationAsRead(userId: string, otherUserId: string) {
    await this.messageRepo
      .createQueryBuilder()
      .update(MessageEntity)
      .set({ isRead: true })
      .where('recipientId = :userId AND senderId = :otherUserId AND isRead = false', {
        userId,
        otherUserId
      })
      .execute();

    return { success: true };
  }
}
