/**
 * Property-Based Tests for Listings Module
 * Feature: maritime-app-enhancements
 * 
 * These tests use fast-check for property-based testing to verify
 * universal properties that should hold across all inputs.
 * 
 * Setup required:
 * npm install --save-dev fast-check @types/jest
 */

import * as fc from 'fast-check';
import { Test, TestingModule } from '@nestjs/testing';
import { ListingsService } from './listings.service';
import { ListingsController } from './listings.controller';
import { getRepositoryToken } from '@nestjs/typeorm';
import { ListingEntity } from './entities/listing.entity';

describe('Listings Property-Based Tests', () => {
  let service: ListingsService;
  let controller: ListingsController;
  let mockRepository: any;

  beforeEach(async () => {
    mockRepository = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn(),
      save: jest.fn(),
      update: jest.fn(),
      delete: jest.fn(),
    };

    const module: TestingModule = await Test.createTestingModule({
      controllers: [ListingsController],
      providers: [
        ListingsService,
        {
          provide: getRepositoryToken(ListingEntity),
          useValue: mockRepository,
        },
      ],
    }).compile();

    service = module.get<ListingsService>(ListingsService);
    controller = module.get<ListingsController>(ListingsController);
  });

  /**
   * Property 26: New listing appears in listings screen
   * Feature: maritime-app-enhancements, Property 26
   * Validates: Requirements 9.3
   * 
   * For any newly created asset listing, it should immediately appear in the Listings screen
   */
  describe('Property 26: New listing appears in listings screen', () => {
    it('should include newly created listing in subsequent fetch', async () => {
      await fc.assert(
        fc.asyncProperty(
          // Generate random listing data
          fc.record({
            name: fc.string({ minLength: 1, maxLength: 100 }),
            assetType: fc.constantFrom('vessel', 'equipment', 'service', 'other'),
            description: fc.string({ minLength: 1, maxLength: 500 }),
            specifications: fc.dictionary(fc.string(), fc.string()),
          }),
          async (listingData) => {
            // Arrange: Mock initial listings
            const initialListings = [
              { id: '1', name: 'Existing Listing 1', assetType: 'vessel' },
              { id: '2', name: 'Existing Listing 2', assetType: 'equipment' },
            ];
            
            mockRepository.find.mockResolvedValueOnce(initialListings);
            
            // Act: Create new listing
            const newListing = {
              id: '3',
              ...listingData,
              company: { id: 'company-1' },
              isActive: true,
              createdAt: new Date(),
              updatedAt: new Date(),
            };
            
            mockRepository.create.mockReturnValue(newListing);
            mockRepository.save.mockResolvedValue(newListing);
            
            // Create the listing
            const created = await service.create(listingData, { id: 'company-1' } as any);
            
            // Mock the updated listings array
            const updatedListings = [...initialListings, created];
            mockRepository.find.mockResolvedValueOnce(updatedListings);
            
            // Fetch all listings
            const allListings = await service.findAll({});
            
            // Assert: New listing should be in the results
            const listingIds = allListings.map((l: any) => l.id);
            expect(listingIds).toContain(created.id);
            expect(allListings.length).toBe(initialListings.length + 1);
          }
        ),
        { numRuns: 100 } // Run 100 iterations as per design spec
      );
    });

    it('should preserve all required fields in created listing', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.record({
            name: fc.string({ minLength: 1, maxLength: 100 }),
            assetType: fc.constantFrom('vessel', 'equipment', 'service', 'other'),
            description: fc.string({ minLength: 1, maxLength: 500 }),
            specifications: fc.dictionary(fc.string(), fc.string()),
          }),
          async (listingData) => {
            // Arrange
            const newListing = {
              id: fc.uuid().generate(fc.random()),
              ...listingData,
              company: { id: 'company-1' },
              isActive: true,
              createdAt: new Date(),
              updatedAt: new Date(),
            };
            
            mockRepository.create.mockReturnValue(newListing);
            mockRepository.save.mockResolvedValue(newListing);
            mockRepository.findOne.mockResolvedValue(newListing);
            
            // Act: Create and fetch
            const created = await service.create(listingData, { id: 'company-1' } as any);
            const fetched = await service.findOne(created.id);
            
            // Assert: All fields should match
            expect(fetched.name).toBe(listingData.name);
            expect(fetched.assetType).toBe(listingData.assetType);
            expect(fetched.description).toBe(listingData.description);
            expect(fetched.specifications).toEqual(listingData.specifications);
          }
        ),
        { numRuns: 100 }
      );
    });
  });

  /**
   * Property 28: Listing deletion removes from display
   * Feature: maritime-app-enhancements, Property 28
   * Validates: Requirements 9.7
   * 
   * For any asset listing, after deletion, it should no longer appear in the Listings screen
   * or exist in the database
   */
  describe('Property 28: Listing deletion removes from display', () => {
    it('should not include deleted listing in subsequent fetch', async () => {
      await fc.assert(
        fc.asyncProperty(
          // Generate random listing to delete
          fc.record({
            id: fc.uuid(),
            name: fc.string({ minLength: 1, maxLength: 100 }),
            assetType: fc.constantFrom('vessel', 'equipment', 'service', 'other'),
          }),
          async (listingToDelete) => {
            // Arrange: Mock listings including the one to delete
            const allListings = [
              { id: '1', name: 'Listing 1', assetType: 'vessel' },
              listingToDelete,
              { id: '3', name: 'Listing 3', assetType: 'equipment' },
            ];
            
            mockRepository.find.mockResolvedValueOnce(allListings);
            mockRepository.findOne.mockResolvedValue(listingToDelete);
            mockRepository.delete.mockResolvedValue({ affected: 1 });
            
            // Act: Delete the listing
            await service.remove(listingToDelete.id, { id: 'company-1' } as any);
            
            // Mock the updated listings array (without deleted listing)
            const remainingListings = allListings.filter(l => l.id !== listingToDelete.id);
            mockRepository.find.mockResolvedValueOnce(remainingListings);
            
            // Fetch all listings
            const fetchedListings = await service.findAll({});
            
            // Assert: Deleted listing should not be in results
            const listingIds = fetchedListings.map((l: any) => l.id);
            expect(listingIds).not.toContain(listingToDelete.id);
            expect(fetchedListings.length).toBe(allListings.length - 1);
          }
        ),
        { numRuns: 100 }
      );
    });

    it('should return null when fetching deleted listing by ID', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.uuid(),
          async (listingId) => {
            // Arrange: Mock listing exists
            const listing = {
              id: listingId,
              name: 'Test Listing',
              assetType: 'vessel',
              company: { id: 'company-1' },
            };
            
            mockRepository.findOne.mockResolvedValueOnce(listing);
            mockRepository.delete.mockResolvedValue({ affected: 1 });
            
            // Act: Delete the listing
            await service.remove(listingId, { id: 'company-1' } as any);
            
            // Mock that listing no longer exists
            mockRepository.findOne.mockResolvedValueOnce(null);
            
            // Try to fetch deleted listing
            const fetched = await service.findOne(listingId);
            
            // Assert: Should return null
            expect(fetched).toBeNull();
          }
        ),
        { numRuns: 100 }
      );
    });

    it('should maintain referential integrity after deletion', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.array(
            fc.record({
              id: fc.uuid(),
              name: fc.string({ minLength: 1, maxLength: 100 }),
              assetType: fc.constantFrom('vessel', 'equipment', 'service', 'other'),
            }),
            { minLength: 2, maxLength: 10 }
          ),
          async (listings) => {
            // Arrange: Pick a random listing to delete
            const indexToDelete = Math.floor(Math.random() * listings.length);
            const listingToDelete = listings[indexToDelete];
            
            mockRepository.find.mockResolvedValueOnce(listings);
            mockRepository.findOne.mockResolvedValue(listingToDelete);
            mockRepository.delete.mockResolvedValue({ affected: 1 });
            
            // Act: Delete one listing
            await service.remove(listingToDelete.id, { id: 'company-1' } as any);
            
            // Mock remaining listings
            const remainingListings = listings.filter(l => l.id !== listingToDelete.id);
            mockRepository.find.mockResolvedValueOnce(remainingListings);
            
            // Fetch all listings
            const fetchedListings = await service.findAll({});
            
            // Assert: Count should be reduced by exactly 1
            expect(fetchedListings.length).toBe(listings.length - 1);
            
            // Assert: All remaining listings should still be present
            const remainingIds = remainingListings.map(l => l.id);
            const fetchedIds = fetchedListings.map((l: any) => l.id);
            expect(fetchedIds.sort()).toEqual(remainingIds.sort());
          }
        ),
        { numRuns: 100 }
      );
    });
  });
});
