import { Test, TestingModule } from '@nestjs/testing';
import { getRepositoryToken } from '@nestjs/typeorm';
import { JobsService } from './jobs.service';
import { JobEntity } from './job.entity';
import { ApplicationEntity } from './application.entity';
import { UserEntity } from '../users/user.entity';
import * as fc from 'fast-check';

/**
 * Feature: maritime-app-enhancements
 * Property 10: New application initial status
 * Property 11: My applications completeness
 * Validates: Requirements 3.4, 3.5
 */

describe('JobsService Property Tests', () => {
  let service: JobsService;
  let applicationRepo: any;

  beforeEach(async () => {
    const mockJobRepo = {
      find: jest.fn(),
      findOne: jest.fn((opts: any) => Promise.resolve({ id: opts.where.id, isActive: true })),
      create: jest.fn(),
      save: jest.fn(),
      update: jest.fn(),
      delete: jest.fn(),
      createQueryBuilder: jest.fn()
    };

    const mockApplicationRepo = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn((data: any) => data),
      save: jest.fn((data: any) => Promise.resolve({ id: 'app-id', ...data })),
      update: jest.fn(),
      delete: jest.fn()
    };

    const mockUserRepo = {
      findOne: jest.fn()
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        JobsService,
        {
          provide: getRepositoryToken(JobEntity),
          useValue: mockJobRepo
        },
        {
          provide: getRepositoryToken(ApplicationEntity),
          useValue: mockApplicationRepo
        },
        {
          provide: getRepositoryToken(UserEntity),
          useValue: mockUserRepo
        }
      ]
    }).compile();

    service = module.get<JobsService>(JobsService);
    applicationRepo = module.get(getRepositoryToken(ApplicationEntity));
  });

  describe('Property 10: New application initial status', () => {
    it('should create application with pending status', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.uuid(),
          fc.uuid(),
          fc.webUrl(),
          fc.option(fc.string(), { nil: undefined }),
          async (jobId, applicantId, cvUrl, coverLetter) => {
            applicationRepo.findOne.mockResolvedValue(null);

            const result = await service.applyToJob(jobId, applicantId, cvUrl, coverLetter);

            expect(result.status).toBe('pending');
            expect(applicationRepo.save).toHaveBeenCalled();
          }
        ),
        { numRuns: 100 }
      );
    });
  });

  describe('Property 11: My applications completeness', () => {
    it('should return all and only applications for the user', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.uuid(),
          fc.array(fc.record({
            id: fc.uuid(),
            applicantId: fc.uuid(),
            jobId: fc.uuid()
          }), { minLength: 1, maxLength: 10 }),
          async (userId, allApplications) => {
            const userApplications = allApplications.filter(app => app.applicantId === userId);
            
            applicationRepo.find.mockResolvedValue(
              userApplications.map(app => ({
                id: app.id,
                applicant: { id: app.applicantId },
                job: { id: app.jobId }
              }))
            );

            const result = await service.getUserApplications(userId);

            expect(result.length).toBe(userApplications.length);
            result.forEach(app => {
              expect(app.applicant.id).toBe(userId);
            });
          }
        ),
        { numRuns: 50 }
      );
    });
  });

  describe('Property 12: Job search filtering', () => {
    it('should return only jobs matching search criteria', async () => {
      const mockJobRepo = service['jobRepo'];
      
      await fc.assert(
        fc.asyncProperty(
          fc.constantFrom('full-time', 'part-time', 'contract', 'internship'),
          fc.string({ minLength: 3, maxLength: 20 }),
          async (jobType, location) => {
            const matchingJobs = [
              { id: '1', jobType, location, isActive: true },
              { id: '2', jobType, location: location + ' City', isActive: true }
            ];
            
            // Mock the query builder chain
            const mockQueryBuilder = {
              leftJoinAndSelect: jest.fn().mockReturnThis(),
              where: jest.fn().mockReturnThis(),
              andWhere: jest.fn().mockReturnThis(),
              orderBy: jest.fn().mockReturnThis(),
              getMany: jest.fn().mockResolvedValue(matchingJobs)
            };
            
            mockJobRepo.createQueryBuilder = jest.fn().mockReturnValue(mockQueryBuilder);

            const result = await service.findAll({ jobType, location });

            result.forEach((job: any) => {
              expect(job.jobType).toBe(jobType);
              expect(job.isActive).toBe(true);
            });
          }
        ),
        { numRuns: 50 }
      );
    });
  });
});

/**
 * Feature: maritime-app-enhancements, Property 7: Backend job validation
 * Validates: Requirements 5.1, 5.2
 */
describe('Property 7: Backend job validation', () => {
  let service: JobsService;
  let jobRepo: any;
  let userRepo: any;

  beforeEach(async () => {
    const mockJobRepo = {
      create: jest.fn((data: any) => data),
      save: jest.fn((data: any) => Promise.resolve({ id: 'job-id', ...data })),
      findOne: jest.fn()
    };

    const mockApplicationRepo = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn(),
      save: jest.fn()
    };

    const mockUserRepo = {
      findOne: jest.fn()
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        JobsService,
        {
          provide: getRepositoryToken(JobEntity),
          useValue: mockJobRepo
        },
        {
          provide: getRepositoryToken(ApplicationEntity),
          useValue: mockApplicationRepo
        },
        {
          provide: getRepositoryToken(UserEntity),
          useValue: mockUserRepo
        }
      ]
    }).compile();

    service = module.get<JobsService>(JobsService);
    jobRepo = module.get(getRepositoryToken(JobEntity));
    userRepo = module.get(getRepositoryToken(UserEntity));
  });

  it('should reject job creation when required fields are missing', async () => {
    await fc.assert(
      fc.asyncProperty(
        fc.record({
          title: fc.option(fc.string(), { nil: undefined }),
          description: fc.option(fc.string(), { nil: undefined }),
          location: fc.option(fc.string(), { nil: undefined }),
          jobType: fc.constant('full-time'),
          salary: fc.option(fc.string(), { nil: undefined })
        }),
        fc.uuid(),
        async (jobData, companyId) => {
          // Only test cases where at least one required field is missing
          if (!jobData.title || !jobData.description || !jobData.location) {
            userRepo.findOne.mockResolvedValue({ id: companyId });

            await expect(service.create(jobData as any, companyId)).rejects.toThrow();
          }
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should include field-specific error messages for missing fields', async () => {
    await fc.assert(
      fc.asyncProperty(
        fc.uuid(),
        async (companyId) => {
          userRepo.findOne.mockResolvedValue({ id: companyId });

          const invalidJob = {
            title: '',
            description: '',
            location: '',
            jobType: 'full-time'
          };

          try {
            await service.create(invalidJob as any, companyId);
            // Should not reach here
            expect(true).toBe(false);
          } catch (error: any) {
            expect(error.message).toBeDefined();
            expect(error.response?.errors || error.response?.message).toBeDefined();
          }
        }
      ),
      { numRuns: 100 }
    );
  });
});

/**
 * Feature: maritime-app-enhancements, Property 8: Job entity creation
 * Validates: Requirements 5.3, 5.4, 5.5
 */
describe('Property 8: Job entity creation', () => {
  let service: JobsService;
  let jobRepo: any;
  let userRepo: any;

  beforeEach(async () => {
    let savedJobData: any = null;

    const mockJobRepo = {
      create: jest.fn((data: any) => data),
      save: jest.fn((data: any) => {
        savedJobData = { id: 'generated-job-id', ...data };
        return Promise.resolve(savedJobData);
      }),
      findOne: jest.fn(() => Promise.resolve(savedJobData))
    };

    const mockApplicationRepo = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn(),
      save: jest.fn()
    };

    const mockUserRepo = {
      findOne: jest.fn((opts: any) => {
        return Promise.resolve({
          id: opts.where.id,
          name: 'Test Company',
          userType: 'company'
        });
      })
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        JobsService,
        {
          provide: getRepositoryToken(JobEntity),
          useValue: mockJobRepo
        },
        {
          provide: getRepositoryToken(ApplicationEntity),
          useValue: mockApplicationRepo
        },
        {
          provide: getRepositoryToken(UserEntity),
          useValue: mockUserRepo
        }
      ]
    }).compile();

    service = module.get<JobsService>(JobsService);
    jobRepo = module.get(getRepositoryToken(JobEntity));
    userRepo = module.get(getRepositoryToken(UserEntity));
  });

  it('should create job entity with company relationship and return complete object with ID', async () => {
    await fc.assert(
      fc.asyncProperty(
        fc.record({
          title: fc.string({ minLength: 1, maxLength: 100 }),
          description: fc.string({ minLength: 1, maxLength: 500 }),
          location: fc.string({ minLength: 1, maxLength: 100 }),
          salary: fc.option(fc.string(), { nil: undefined }),
          jobType: fc.constantFrom('Full-time', 'Part-time', 'Contract', 'Internship'),
          requirements: fc.option(fc.string(), { nil: undefined }),
          isActive: fc.option(fc.boolean(), { nil: undefined })
        }),
        fc.uuid(),
        async (jobData, companyId) => {
          const result = await service.create(jobData as any, companyId);

          // Verify the job has a generated ID
          expect(result).toBeDefined();
          expect(result?.id).toBeDefined();
          expect(result?.id).toBe('generated-job-id');

          // Verify company relationship is established
          expect(result?.company).toBeDefined();
          expect(result?.company.id).toBe(companyId);

          // Verify all fields are present
          expect(result?.title).toBe(jobData.title);
          expect(result?.description).toBe(jobData.description);
          expect(result?.location).toBe(jobData.location);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should return 404 when company is not found', async () => {
    await fc.assert(
      fc.asyncProperty(
        fc.record({
          title: fc.string({ minLength: 1, maxLength: 100 }),
          description: fc.string({ minLength: 1, maxLength: 500 }),
          location: fc.string({ minLength: 1, maxLength: 100 }),
          jobType: fc.constant('full-time')
        }),
        fc.uuid(),
        async (jobData, companyId) => {
          userRepo.findOne.mockResolvedValue(null);

          await expect(service.create(jobData as any, companyId)).rejects.toThrow('Company not found');
        }
      ),
      { numRuns: 100 }
    );
  });
});

/**
 * Feature: maritime-app-enhancements, Property 10: Job type enum validation
 * Validates: Requirements 5.6
 */
describe('Property 10: Job type enum validation', () => {
  let service: JobsService;
  let jobRepo: any;
  let userRepo: any;

  beforeEach(async () => {
    let savedJobData: any = null;

    const mockJobRepo = {
      create: jest.fn((data: any) => data),
      save: jest.fn((data: any) => {
        savedJobData = { id: 'generated-job-id', ...data };
        return Promise.resolve(savedJobData);
      }),
      findOne: jest.fn(() => Promise.resolve(savedJobData))
    };

    const mockApplicationRepo = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn(),
      save: jest.fn()
    };

    const mockUserRepo = {
      findOne: jest.fn((opts: any) => {
        return Promise.resolve({
          id: opts.where.id,
          name: 'Test Company',
          userType: 'company'
        });
      })
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        JobsService,
        {
          provide: getRepositoryToken(JobEntity),
          useValue: mockJobRepo
        },
        {
          provide: getRepositoryToken(ApplicationEntity),
          useValue: mockApplicationRepo
        },
        {
          provide: getRepositoryToken(UserEntity),
          useValue: mockUserRepo
        }
      ]
    }).compile();

    service = module.get<JobsService>(JobsService);
    jobRepo = module.get(getRepositoryToken(JobEntity));
    userRepo = module.get(getRepositoryToken(UserEntity));
  });

  it('should accept valid job types (case-insensitive)', async () => {
    const validJobTypes = ['full-time', 'part-time', 'contract', 'internship'];
    
    await fc.assert(
      fc.asyncProperty(
        fc.record({
          title: fc.string({ minLength: 1, maxLength: 100 }),
          description: fc.string({ minLength: 1, maxLength: 500 }),
          location: fc.string({ minLength: 1, maxLength: 100 }),
          jobType: fc.constantFrom('Full-time', 'PART-TIME', 'Contract', 'InTeRnShIp', 'full-time')
        }),
        fc.uuid(),
        async (jobData, companyId) => {
          const result = await service.create(jobData as any, companyId);

          // Verify the job type is normalized to lowercase
          expect(result).toBeDefined();
          expect(result?.jobType).toBeDefined();
          expect(validJobTypes).toContain(result?.jobType);
        }
      ),
      { numRuns: 100 }
    );
  });

  it('should reject invalid job types', async () => {
    await fc.assert(
      fc.asyncProperty(
        fc.record({
          title: fc.string({ minLength: 1, maxLength: 100 }),
          description: fc.string({ minLength: 1, maxLength: 500 }),
          location: fc.string({ minLength: 1, maxLength: 100 }),
          jobType: fc.constantFrom('invalid', 'temporary', 'freelance', 'volunteer', 'seasonal')
        }),
        fc.uuid(),
        async (jobData, companyId) => {
          await expect(service.create(jobData as any, companyId)).rejects.toThrow();
        }
      ),
      { numRuns: 100 }
    );
  });
});

/**
 * Unit tests for job creation backend
 * Validates: Requirements 5.1, 5.2, 5.3, 5.4, 5.5, 5.6
 */
describe('JobsService Unit Tests - Job Creation', () => {
  let service: JobsService;
  let jobRepo: any;
  let userRepo: any;

  beforeEach(async () => {
    let savedJobData: any = null;

    const mockJobRepo = {
      create: jest.fn((data: any) => data),
      save: jest.fn((data: any) => {
        savedJobData = { id: 'test-job-id', ...data };
        return Promise.resolve(savedJobData);
      }),
      findOne: jest.fn(() => Promise.resolve(savedJobData))
    };

    const mockApplicationRepo = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn(),
      save: jest.fn()
    };

    const mockUserRepo = {
      findOne: jest.fn()
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        JobsService,
        {
          provide: getRepositoryToken(JobEntity),
          useValue: mockJobRepo
        },
        {
          provide: getRepositoryToken(ApplicationEntity),
          useValue: mockApplicationRepo
        },
        {
          provide: getRepositoryToken(UserEntity),
          useValue: mockUserRepo
        }
      ]
    }).compile();

    service = module.get<JobsService>(JobsService);
    jobRepo = module.get(getRepositoryToken(JobEntity));
    userRepo = module.get(getRepositoryToken(UserEntity));
  });

  it('should successfully create a job with all valid fields', async () => {
    const companyUser = {
      id: 'company-123',
      name: 'Test Company',
      userType: 'company'
    };
    userRepo.findOne.mockResolvedValue(companyUser);

    const jobDto = {
      title: 'Software Engineer',
      description: 'We are looking for a software engineer',
      location: 'New York',
      salary: '$100k-$150k',
      jobType: 'Full-time',
      requirements: 'Bachelor degree in CS',
      isActive: true
    };

    const result = await service.create(jobDto as any, 'company-123');

    expect(result).toBeDefined();
    expect(result?.id).toBe('test-job-id');
    expect(result?.title).toBe(jobDto.title);
    expect(result?.description).toBe(jobDto.description);
    expect(result?.location).toBe(jobDto.location);
    expect(result?.salaryRange).toBe(jobDto.salary);
    expect(result?.jobType).toBe('full-time'); // normalized to lowercase
    expect(result?.requirements).toBe(jobDto.requirements);
    expect(result?.company).toEqual(companyUser);
    expect(jobRepo.create).toHaveBeenCalled();
    expect(jobRepo.save).toHaveBeenCalled();
  });

  it('should return validation error when title is missing', async () => {
    const companyUser = { id: 'company-123', name: 'Test Company' };
    userRepo.findOne.mockResolvedValue(companyUser);

    const jobDto = {
      title: '',
      description: 'Description',
      location: 'Location',
      jobType: 'full-time'
    };

    await expect(service.create(jobDto as any, 'company-123')).rejects.toThrow('Validation failed');
  });

  it('should return validation error when description is missing', async () => {
    const companyUser = { id: 'company-123', name: 'Test Company' };
    userRepo.findOne.mockResolvedValue(companyUser);

    const jobDto = {
      title: 'Title',
      description: '',
      location: 'Location',
      jobType: 'full-time'
    };

    await expect(service.create(jobDto as any, 'company-123')).rejects.toThrow('Validation failed');
  });

  it('should return validation error when location is missing', async () => {
    const companyUser = { id: 'company-123', name: 'Test Company' };
    userRepo.findOne.mockResolvedValue(companyUser);

    const jobDto = {
      title: 'Title',
      description: 'Description',
      location: '',
      jobType: 'full-time'
    };

    await expect(service.create(jobDto as any, 'company-123')).rejects.toThrow('Validation failed');
  });

  it('should establish company relationship with UserEntity', async () => {
    const companyUser = {
      id: 'company-456',
      name: 'Another Company',
      userType: 'company'
    };
    userRepo.findOne.mockResolvedValue(companyUser);

    const jobDto = {
      title: 'Data Analyst',
      description: 'Analyze data',
      location: 'Remote',
      jobType: 'contract'
    };

    const result = await service.create(jobDto as any, 'company-456');

    expect(result?.company).toEqual(companyUser);
    expect(userRepo.findOne).toHaveBeenCalledWith({ where: { id: 'company-456' } });
  });

  it('should validate jobType against enum values', async () => {
    const companyUser = { id: 'company-123', name: 'Test Company' };
    userRepo.findOne.mockResolvedValue(companyUser);

    const jobDto = {
      title: 'Title',
      description: 'Description',
      location: 'Location',
      jobType: 'invalid-type'
    };

    await expect(service.create(jobDto as any, 'company-123')).rejects.toThrow('Validation failed');
  });

  it('should normalize jobType to lowercase', async () => {
    const companyUser = { id: 'company-123', name: 'Test Company' };
    userRepo.findOne.mockResolvedValue(companyUser);

    const jobDto = {
      title: 'Title',
      description: 'Description',
      location: 'Location',
      jobType: 'FULL-TIME'
    };

    const result = await service.create(jobDto as any, 'company-123');

    expect(result?.jobType).toBe('full-time');
  });

  it('should use default values for optional fields', async () => {
    const companyUser = { id: 'company-123', name: 'Test Company' };
    userRepo.findOne.mockResolvedValue(companyUser);

    const jobDto = {
      title: 'Title',
      description: 'Description',
      location: 'Location',
      jobType: 'part-time'
    };

    const result = await service.create(jobDto as any, 'company-123');

    expect(result?.salaryRange).toBe('Not specified');
    expect(result?.requirements).toBe('');
    expect(result?.isActive).toBe(true);
  });
});
