import { Body, Controller, Delete, Get, Param, Patch, Post, Query, Req, UseGuards, ValidationPipe, UsePipes, UseInterceptors, UploadedFile, BadRequestException } from '@nestjs/common';
import { FileInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { extname } from 'path';
import { JwtAuthGuard } from '../auth/jwt-auth.guard';
import { JobsService } from './jobs.service';
import { CreateJobDto } from './dto/create-job.dto';
import { UpdateJobDto } from './dto/update-job.dto';

@Controller('jobs')
@UseGuards(JwtAuthGuard)
export class JobsController {
  constructor(private readonly jobsService: JobsService) {}

  @Get()
  getJobs(@Query() filters: any) {
    return this.jobsService.findAll(filters);
  }

  @Get('applications')
  getUserApplications(@Req() req: any) {
    return this.jobsService.getUserApplications(req.user.userId);
  }

  @Get(':id')
  getJob(@Param('id') id: string) {
    return this.jobsService.findOne(id);
  }

  @Get(':id/applications')
  getJobApplications(@Param('id') jobId: string) {
    return this.jobsService.getJobApplications(jobId);
  }

  @Post()
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  create(@Req() req: any, @Body() createJobDto: CreateJobDto) {
    // Pass the DTO and company ID from authenticated user
    return this.jobsService.create(createJobDto, req.user.userId);
  }

  @Post(':id/apply')
  @UseInterceptors(FileInterceptor('cv', {
    storage: diskStorage({
      destination: './uploads/cvs',
      filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        const ext = extname(file.originalname);
        cb(null, `cv-${uniqueSuffix}${ext}`);
      }
    }),
    fileFilter: (req, file, cb) => {
      // Accept PDF, DOC, and DOCX files
      if (file.mimetype === 'application/pdf' || 
          file.mimetype === 'application/msword' || 
          file.mimetype === 'application/vnd.openxmlformats-officedocument.wordprocessingml.document') {
        cb(null, true);
      } else {
        cb(new BadRequestException('Only PDF, DOC, and DOCX files are allowed'), false);
      }
    },
    limits: {
      fileSize: 5 * 1024 * 1024 // 5MB limit
    }
  }))
  applyToJob(
    @Req() req: any, 
    @Param('id') jobId: string, 
    @UploadedFile() cv: Express.Multer.File,
    @Body() body: { coverLetter?: string }
  ) {
    console.log('Apply to job endpoint hit');
    console.log('Job ID:', jobId);
    console.log('User ID:', req.user?.userId);
    console.log('CV file:', cv);
    console.log('Body:', body);
    
    if (!cv) {
      console.error('No CV file received');
      throw new BadRequestException('CV file is required');
    }
    
    console.log('CV filename:', cv.filename);
    const cvUrl = `/uploads/cvs/${cv.filename}`;
    return this.jobsService.applyToJob(jobId, req.user.userId, cvUrl, body.coverLetter);
  }

  @Patch(':id')
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  update(@Param('id') id: string, @Body() updateJobDto: UpdateJobDto) {
    return this.jobsService.update(id, updateJobDto);
  }

  @Patch('applications/:id')
  updateApplicationStatus(@Param('id') applicationId: string, @Body() body: { status: string }) {
    return this.jobsService.updateApplicationStatus(applicationId, body.status as any);
  }

  @Delete(':id')
  remove(@Param('id') id: string) {
    return this.jobsService.remove(id);
  }
}


