import { Injectable, UnauthorizedException, BadRequestException, NotFoundException } from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import * as bcrypt from 'bcryptjs';
import { UsersService } from '../users/users.service';
import { LoginDto } from './dto/login.dto';
import { RegisterDto } from './dto/register.dto';
import { VerifyEmailDto } from './dto/verify-email.dto';
import { ForgotPasswordDto } from './dto/forgot-password.dto';
import { ResetPasswordDto } from './dto/reset-password.dto';
import { ResendOtpDto } from './dto/resend-otp.dto';
import { EmailService } from './email.service';

@Injectable()
export class AuthService {
  constructor(
    private readonly usersService: UsersService, 
    private readonly jwtService: JwtService,
    private readonly emailService: EmailService
  ) {}

  async login(dto: LoginDto) {
    const user = await this.usersService.findByEmail(dto.email);

    // Fallback to mock accounts matching your RN demo if no DB user
    const mockCredentials: Record<string, string> = {
      'captain@maritime.com': 'Captain123!',
      'engineer@shipping.com': 'Engineer456!',
      'port@manager.com': 'Port789!'
    };

    let valid = false;
    let userId = user?.id;
    let userType = user?.userType || 'professional';
    let name = user?.name || dto.email.split('@')[0];
    let isEmailVerified = user?.isEmailVerified ?? true; // Mock accounts are considered verified

    if (user?.passwordHash) {
      valid = await bcrypt.compare(dto.password, user.passwordHash);
      
      // Check if email is verified for real users
      if (valid && !user.isEmailVerified) {
        throw new UnauthorizedException('Please verify your email before logging in. Check your inbox for the verification code.');
      }
    } else if (mockCredentials[dto.email]) {
      valid = mockCredentials[dto.email] === dto.password;
      userId = userId || '1';
      isEmailVerified = true; // Mock accounts are always verified
    }

    if (!valid) {
      throw new UnauthorizedException('Invalid maritime credentials. Please check your email and password.');
    }

    const payload = { sub: userId, email: dto.email, userType };
    const token = await this.jwtService.signAsync(payload);

    return {
      user: { id: userId, email: dto.email, name, userType, isEmailVerified },
      token
    };
  }

  async register(dto: RegisterDto) {
    // Check if user already exists
    const existingUser = await this.usersService.findByEmail(dto.email);
    if (existingUser) {
      throw new BadRequestException('User with this email already exists');
    }

    const passwordHash = await bcrypt.hash(dto.password, 10);
    
    // Generate OTP for email verification
    const otp = this.generateOTP();
    const otpExpires = new Date(Date.now() + 10 * 60 * 1000); // 10 minutes

    const created = await this.usersService.createUser({
      email: dto.email,
      name: dto.name,
      passwordHash,
      userType: dto.userType,
      isEmailVerified: false,
      emailVerificationToken: otp,
      emailVerificationExpires: otpExpires
    });

    // Send verification email
    await this.emailService.sendVerificationEmail(dto.email, otp);

    return {
      user: { 
        id: created.id, 
        email: created.email, 
        name: created.name, 
        userType: created.userType,
        isEmailVerified: false
      },
      message: 'Registration successful. Please check your email for verification code.'
    };
  }

  async verifyEmail(dto: VerifyEmailDto) {
    const user = await this.usersService.findByEmail(dto.email);
    if (!user) {
      throw new NotFoundException('User not found');
    }

    if (user.isEmailVerified) {
      throw new BadRequestException('Email is already verified');
    }

    if (!user.emailVerificationToken || user.emailVerificationToken !== dto.otp) {
      throw new BadRequestException('Invalid verification code');
    }

    if (!user.emailVerificationExpires || user.emailVerificationExpires < new Date()) {
      throw new BadRequestException('Verification code has expired');
    }

    // Update user as verified
    await this.usersService.updateUser(user.id, {
      isEmailVerified: true,
      emailVerificationToken: undefined,
      emailVerificationExpires: undefined
    });

    // Generate JWT token for authenticated session
    const payload = { sub: user.id, email: user.email, userType: user.userType };
    const token = await this.jwtService.signAsync(payload);

    return {
      user: { 
        id: user.id, 
        email: user.email, 
        name: user.name, 
        userType: user.userType,
        isEmailVerified: true
      },
      token,
      message: 'Email verified successfully'
    };
  }

  async resendVerificationOtp(dto: ResendOtpDto) {
    const user = await this.usersService.findByEmail(dto.email);
    if (!user) {
      throw new NotFoundException('User not found');
    }

    if (user.isEmailVerified) {
      throw new BadRequestException('Email is already verified');
    }

    // Generate new OTP
    const otp = this.generateOTP();
    const otpExpires = new Date(Date.now() + 10 * 60 * 1000); // 10 minutes

    await this.usersService.updateUser(user.id, {
      emailVerificationToken: otp,
      emailVerificationExpires: otpExpires
    });

    // Send verification email
    await this.emailService.sendVerificationEmail(dto.email, otp);

    return {
      message: 'Verification code sent successfully'
    };
  }

  async forgotPassword(dto: ForgotPasswordDto) {
    const user = await this.usersService.findByEmail(dto.email);
    if (!user) {
      // Don't reveal if user exists or not for security
      return {
        message: 'If an account with this email exists, you will receive a password reset code.'
      };
    }

    // Generate OTP for password reset
    const otp = this.generateOTP();
    const otpExpires = new Date(Date.now() + 10 * 60 * 1000); // 10 minutes

    await this.usersService.updateUser(user.id, {
      passwordResetToken: otp,
      passwordResetExpires: otpExpires
    });

    // Send password reset email
    await this.emailService.sendPasswordResetEmail(dto.email, otp);

    return {
      message: 'If an account with this email exists, you will receive a password reset code.'
    };
  }

  async resetPassword(dto: ResetPasswordDto) {
    const user = await this.usersService.findByEmail(dto.email);
    if (!user) {
      throw new NotFoundException('User not found');
    }

    if (!user.passwordResetToken || user.passwordResetToken !== dto.otp) {
      throw new BadRequestException('Invalid reset code');
    }

    if (!user.passwordResetExpires || user.passwordResetExpires < new Date()) {
      throw new BadRequestException('Reset code has expired');
    }

    // Hash new password
    const passwordHash = await bcrypt.hash(dto.newPassword, 10);

    // Update user password and clear reset tokens
    await this.usersService.updateUser(user.id, {
      passwordHash,
      passwordResetToken: undefined,
      passwordResetExpires: undefined
    });

    return {
      message: 'Password reset successfully'
    };
  }

  private generateOTP(): string {
    return Math.floor(100000 + Math.random() * 900000).toString();
  }
}


