import { Test, TestingModule } from '@nestjs/testing';
import { JwtService } from '@nestjs/jwt';
import { AuthService } from './auth.service';
import { UsersService } from '../users/users.service';
import * as fc from 'fast-check';
import { validate } from 'class-validator';
import { RegisterDto } from './dto/register.dto';

/**
 * Feature: maritime-app-enhancements, Property 13: Registration field validation
 * Validates: Requirements 4.4
 * 
 * For any registration attempt, if required fields are missing, 
 * the system should reject the registration and display validation errors
 */

describe('AuthService Property Tests', () => {
  let service: AuthService;
  let usersService: UsersService;
  let jwtService: JwtService;

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        AuthService,
        {
          provide: UsersService,
          useValue: {
            findByEmail: jest.fn(),
            createUser: jest.fn()
          }
        },
        {
          provide: JwtService,
          useValue: {
            signAsync: jest.fn().mockResolvedValue('mock-token')
          }
        }
      ]
    }).compile();

    service = module.get<AuthService>(AuthService);
    usersService = module.get<UsersService>(UsersService);
    jwtService = module.get<JwtService>(JwtService);
  });

  describe('Property 13: Registration field validation', () => {
    it('should reject registration with missing required fields', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.record({
            email: fc.oneof(
              fc.constant(''),
              fc.constant(undefined as any),
              fc.string().filter(s => !s.includes('@'))
            ),
            password: fc.oneof(
              fc.constant(''),
              fc.constant(undefined as any),
              fc.string().filter(s => s.length < 6)
            ),
            name: fc.option(fc.string(), { nil: undefined }),
            role: fc.option(fc.string(), { nil: undefined })
          }),
          async (invalidData) => {
            const dto = new RegisterDto();
            Object.assign(dto, invalidData);

            const errors = await validate(dto);
            
            // If email is invalid or password is invalid, there should be validation errors
            const hasInvalidEmail = !invalidData.email || !invalidData.email.includes('@');
            const hasInvalidPassword = !invalidData.password || invalidData.password.length < 6;
            
            if (hasInvalidEmail || hasInvalidPassword) {
              expect(errors.length).toBeGreaterThan(0);
            }
          }
        ),
        { numRuns: 100 }
      );
    });

    it('should accept registration with all required fields valid', async () => {
      await fc.assert(
        fc.asyncProperty(
          fc.record({
            email: fc.emailAddress(),
            password: fc.string({ minLength: 6, maxLength: 20 }),
            name: fc.option(fc.string({ minLength: 1, maxLength: 50 }), { nil: undefined }),
            role: fc.option(fc.constantFrom('professional', 'company'), { nil: undefined })
          }),
          async (validData) => {
            const dto = new RegisterDto();
            Object.assign(dto, validData);

            const errors = await validate(dto);
            
            // With valid email and password, there should be no validation errors
            expect(errors.length).toBe(0);
          }
        ),
        { numRuns: 100 }
      );
    });
  });
});
