"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("reflect-metadata");
const core_1 = require("@nestjs/core");
const app_module_1 = require("./app.module");
const typeorm_1 = require("typeorm");
const user_entity_1 = require("./users/user.entity");
const profile_entity_1 = require("./profile/profile.entity");
const post_entity_1 = require("./posts/post.entity");
const likes_entity_1 = require("./posts/likes.entity");
const job_entity_1 = require("./jobs/job.entity");
const news_entity_1 = require("./news/news.entity");
const vessel_entity_1 = require("./vessels/vessel.entity");
const connection_entity_1 = require("./network/connection.entity");
const message_entity_1 = require("./messages/message.entity");
const bcrypt = require("bcryptjs");
const fs_1 = require("fs");
async function ensureUploadDirs() {
    const dirs = ["uploads", "uploads/avatars", "uploads/news", "uploads/posts"];
    dirs.forEach((d) => {
        if (!(0, fs_1.existsSync)(d))
            (0, fs_1.mkdirSync)(d);
    });
}
async function bootstrap() {
    await ensureUploadDirs();
    const app = await core_1.NestFactory.createApplicationContext(app_module_1.AppModule);
    const dataSource = app.get(typeorm_1.DataSource);
    const userRepo = dataSource.getRepository(user_entity_1.UserEntity);
    const profileRepo = dataSource.getRepository(profile_entity_1.ProfileEntity);
    const postRepo = dataSource.getRepository(post_entity_1.PostEntity);
    const jobRepo = dataSource.getRepository(job_entity_1.JobEntity);
    const newsRepo = dataSource.getRepository(news_entity_1.NewsEntity);
    const vesselRepo = dataSource.getRepository(vessel_entity_1.VesselEntity);
    const connectionRepo = dataSource.getRepository(connection_entity_1.ConnectionEntity);
    const messageRepo = dataSource.getRepository(message_entity_1.MessageEntity);
    const likesRepo = dataSource.getRepository(likes_entity_1.LikesEntity);
    await messageRepo.delete({});
    await connectionRepo.delete({});
    await postRepo.delete({});
    await newsRepo.delete({});
    await jobRepo.delete({});
    await vesselRepo.delete({});
    await profileRepo.delete({});
    await userRepo.delete({});
    const users = [
        {
            email: "captain@maritime.com",
            name: "Captain John",
            role: "Captain",
            password: "Captain123!",
        },
        {
            email: "engineer@shipping.com",
            name: "Chief Engineer",
            role: "Engineer",
            password: "Engineer456!",
        },
        {
            email: "port@manager.com",
            name: "Port Manager",
            role: "Manager",
            password: "Port789!",
        },
    ];
    const createdUsers = [];
    for (const u of users) {
        const passwordHash = await bcrypt.hash(u.password, 10);
        const user = userRepo.create({
            email: u.email,
            name: u.name,
            userType: 'professional',
            passwordHash,
        });
        createdUsers.push(await userRepo.save(user));
    }
    for (const u of createdUsers) {
        const profile = profileRepo.create({
            user: u,
            currentCompany: u.userType === 'professional' ? "Maritime Shipping Co." : undefined,
            experience: "10 years",
            certifications: ["STCW"],
            skills: ["Navigation", "Safety Management"],
        });
        await profileRepo.save(profile);
    }
    for (const u of createdUsers) {
        const post = postRepo.create({
            author: u,
            content: `Hello from ${u.name}!`,
        });
        const savedPost = await postRepo.save(post);
        const randomLikeCount = Math.floor(Math.random() * 20);
        const randomUsers = createdUsers
            .sort(() => 0.5 - Math.random())
            .slice(0, randomLikeCount);
        const likes = randomUsers.map((user) => likesRepo.create({
            user: user,
            post: savedPost,
        }));
        await likesRepo.save(likes);
    }
    const job1 = jobRepo.create({
        title: "Chief Engineer",
        description: "Experienced chief engineer needed for maritime operations",
        location: "Singapore",
        salaryRange: "$8,000 - $12,000",
        jobType: "full-time",
        requirements: "10+ years experience, STCW certification",
        company: createdUsers[0],
    });
    await jobRepo.save(job1);
    const job2 = jobRepo.create({
        title: "Vessel Captain",
        description: "Seeking experienced vessel captain",
        location: "Rotterdam",
        salaryRange: "$10,000 - $15,000",
        jobType: "full-time",
        requirements: "Master's license, 15+ years experience",
        company: createdUsers[0],
    });
    await jobRepo.save(job2);
    await newsRepo.save(newsRepo.create({
        title: "New Maritime Regulations Coming",
        summary: "Important updates to international shipping regulations...",
        author: "Maritime News",
    }));
    await vesselRepo.save(vesselRepo.create({
        name: "MV Ocean Star",
        type: "Container Ship",
        status: "Available",
        capacity: "5000 TEU",
        location: "Singapore Port",
    }));
    await connectionRepo.save(connectionRepo.create({
        user: createdUsers[0],
        connection: createdUsers[1],
        status: "accepted",
    }));
    const message = messageRepo.create({
        sender: createdUsers[1],
        recipient: createdUsers[0],
        content: "Hello! How are you?",
    });
    await messageRepo.save(message);
    console.log("Seed completed.");
    await app.close();
}
bootstrap().catch((e) => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=seed.js.map