"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MigrationValidator = void 0;
const job_entity_1 = require("../src/jobs/job.entity");
class MigrationValidator {
    constructor(dataSource) {
        this.dataSource = dataSource;
    }
    async validateSchema() {
        try {
            const queryRunner = this.dataSource.createQueryRunner();
            const columns = await queryRunner.getTable('jobs');
            const columnNames = (columns === null || columns === void 0 ? void 0 : columns.columns.map(col => col.name)) || [];
            const requiredColumns = ['minSalary', 'maxSalary', 'currency', 'salaryDuration'];
            const missingColumns = requiredColumns.filter(col => !columnNames.includes(col));
            if (missingColumns.length > 0) {
                return {
                    success: false,
                    message: `Missing required columns: ${missingColumns.join(', ')}`,
                    details: { missingColumns, existingColumns: columnNames }
                };
            }
            const salaryColumns = columns === null || columns === void 0 ? void 0 : columns.columns.filter(col => ['minSalary', 'maxSalary'].includes(col.name));
            for (const col of salaryColumns || []) {
                if (col.type !== 'numeric' && col.type !== 'decimal') {
                    return {
                        success: false,
                        message: `Invalid column type for ${col.name}: expected decimal, got ${col.type}`,
                        details: { column: col.name, type: col.type }
                    };
                }
            }
            await queryRunner.release();
            return {
                success: true,
                message: 'Schema validation passed - all required columns exist with correct types'
            };
        }
        catch (error) {
            return {
                success: false,
                message: `Schema validation failed: ${error instanceof Error ? error.message : String(error)}`,
                details: error
            };
        }
    }
    async validateDataIntegrity() {
        try {
            const jobRepository = this.dataSource.getRepository(job_entity_1.JobEntity);
            const totalJobs = await jobRepository.count();
            const jobsWithLegacySalary = await jobRepository.count({
                where: { salaryRange: { $ne: null } }
            });
            const invalidSalaryRanges = await jobRepository
                .createQueryBuilder('job')
                .where('job.minSalary IS NOT NULL')
                .andWhere('job.maxSalary IS NOT NULL')
                .andWhere('job.minSalary > job.maxSalary')
                .getCount();
            if (invalidSalaryRanges > 0) {
                return {
                    success: false,
                    message: `Found ${invalidSalaryRanges} jobs with invalid salary ranges (min > max)`,
                    details: { invalidRanges: invalidSalaryRanges }
                };
            }
            return {
                success: true,
                message: 'Data integrity validation passed',
                details: {
                    totalJobs,
                    jobsWithLegacySalary,
                    invalidSalaryRanges
                }
            };
        }
        catch (error) {
            return {
                success: false,
                message: `Data integrity validation failed: ${error instanceof Error ? error.message : String(error)}`,
                details: error
            };
        }
    }
    async validateEnumConstraints() {
        try {
            const queryRunner = this.dataSource.createQueryRunner();
            try {
                await queryRunner.query(`
          INSERT INTO jobs (id, title, description, location, "jobType", requirements, currency, "companyId")
          VALUES ('test-currency-uuid', 'Test Job', 'Test Description', 'Test Location', 'full-time', 'Test Requirements', 'INVALID_CURRENCY', 'test-company-uuid')
        `);
                await queryRunner.query(`DELETE FROM jobs WHERE id = 'test-currency-uuid'`);
                return {
                    success: false,
                    message: 'Currency enum constraint is not working - invalid currency was accepted',
                    details: { constraint: 'currency enum' }
                };
            }
            catch (error) {
                if (error instanceof Error && error.message.includes('invalid input value for enum')) {
                }
                else {
                    throw error;
                }
            }
            try {
                await queryRunner.query(`
          INSERT INTO jobs (id, title, description, location, "jobType", requirements, "salaryDuration", "companyId")
          VALUES ('test-duration-uuid', 'Test Job', 'Test Description', 'Test Location', 'full-time', 'Test Requirements', 'invalid_duration', 'test-company-uuid')
        `);
                await queryRunner.query(`DELETE FROM jobs WHERE id = 'test-duration-uuid'`);
                return {
                    success: false,
                    message: 'Duration enum constraint is not working - invalid duration was accepted',
                    details: { constraint: 'salaryDuration enum' }
                };
            }
            catch (error) {
                if (error instanceof Error && error.message.includes('invalid input value for enum')) {
                }
                else {
                    throw error;
                }
            }
            await queryRunner.release();
            return {
                success: true,
                message: 'Enum constraints validation passed - invalid values are properly rejected'
            };
        }
        catch (error) {
            return {
                success: false,
                message: `Enum constraints validation failed: ${error instanceof Error ? error.message : String(error)}`,
                details: error
            };
        }
    }
    async runAllValidations() {
        console.log('🔍 Running migration validation tests...\n');
        const results = [];
        console.log('📋 Validating database schema...');
        const schemaResult = await this.validateSchema();
        results.push(schemaResult);
        console.log(schemaResult.success ? '✅' : '❌', schemaResult.message);
        console.log('\n📊 Validating data integrity...');
        const dataResult = await this.validateDataIntegrity();
        results.push(dataResult);
        console.log(dataResult.success ? '✅' : '❌', dataResult.message);
        if (dataResult.details) {
            console.log('   Details:', dataResult.details);
        }
        console.log('\n🔒 Validating enum constraints...');
        const enumResult = await this.validateEnumConstraints();
        results.push(enumResult);
        console.log(enumResult.success ? '✅' : '❌', enumResult.message);
        return results;
    }
}
exports.MigrationValidator = MigrationValidator;
if (require.main === module) {
    async function main() {
        const { AppDataSource } = await Promise.resolve().then(() => require('../src/data-source'));
        try {
            await AppDataSource.initialize();
            console.log('📡 Connected to database\n');
            const validator = new MigrationValidator(AppDataSource);
            const results = await validator.runAllValidations();
            const allPassed = results.every(result => result.success);
            console.log('\n' + '='.repeat(50));
            console.log(allPassed ? '🎉 All validations passed!' : '💥 Some validations failed!');
            console.log('='.repeat(50));
            if (!allPassed) {
                const failures = results.filter(r => !r.success);
                console.log('\n❌ Failed validations:');
                failures.forEach(failure => {
                    console.log(`   - ${failure.message}`);
                });
                process.exit(1);
            }
        }
        catch (error) {
            console.error('💥 Validation script failed:', error instanceof Error ? error.message : String(error));
            process.exit(1);
        }
        finally {
            await AppDataSource.destroy();
        }
    }
    main();
}
//# sourceMappingURL=validate-migration.js.map