# Job Salary Enhancement API Documentation

## Overview

The Job Salary Enhancement feature introduces structured salary information to replace the simple string-based `salaryRange` field. This enhancement provides better standardization and comparison capabilities for job seekers.

## API Changes

### Job Entity Structure

#### Enhanced Job Object
```json
{
  "id": "uuid",
  "title": "string",
  "description": "string",
  "location": "string",
  "jobType": "full-time | part-time | contract | internship",
  "requirements": "string",
  "company": "UserEntity",
  "isActive": "boolean",
  "postedDate": "ISO 8601 date",
  "updatedAt": "ISO 8601 date",
  
  // Legacy field (maintained for backward compatibility)
  "salaryRange": "string | null",
  
  // New structured salary fields
  "minSalary": "number | null",
  "maxSalary": "number | null", 
  "currency": "string | null",
  "salaryDuration": "string | null"
}
```

#### Salary Field Specifications

| Field | Type | Required | Description | Example |
|-------|------|----------|-------------|---------|
| `minSalary` | number | No | Minimum salary amount | 50000 |
| `maxSalary` | number | No | Maximum salary amount | 75000 |
| `currency` | enum | No | ISO 4217 currency code | "USD" |
| `salaryDuration` | enum | No | Salary time period | "annually" |

#### Supported Currency Codes
```
USD, EUR, GBP, CAD, AUD, JPY, CHF, SEK, NOK, DKK
```

#### Supported Duration Types
```
hourly, daily, weekly, monthly, annually
```

## API Endpoints

### Create Job
**POST** `/jobs`

#### Request Body
```json
{
  "title": "Senior Maritime Engineer",
  "description": "Join our team as a senior maritime engineer...",
  "location": "Hamburg, Germany",
  "jobType": "full-time",
  "requirements": "5+ years experience in maritime engineering...",
  
  // Optional salary fields
  "minSalary": 65000,
  "maxSalary": 85000,
  "currency": "EUR",
  "salaryDuration": "annually"
}
```

#### Response
```json
{
  "id": "123e4567-e89b-12d3-a456-426614174000",
  "title": "Senior Maritime Engineer",
  "description": "Join our team as a senior maritime engineer...",
  "location": "Hamburg, Germany",
  "jobType": "full-time",
  "requirements": "5+ years experience in maritime engineering...",
  "salaryRange": null,
  "minSalary": 65000,
  "maxSalary": 85000,
  "currency": "EUR",
  "salaryDuration": "annually",
  "company": {
    "id": "company-uuid",
    "name": "Maritime Solutions GmbH",
    "email": "hr@maritime-solutions.de"
  },
  "isActive": true,
  "postedDate": "2024-01-15T10:30:00Z",
  "updatedAt": "2024-01-15T10:30:00Z"
}
```

### Update Job
**PUT** `/jobs/:id`

#### Request Body
```json
{
  "title": "Senior Maritime Engineer (Updated)",
  "minSalary": 70000,
  "maxSalary": 90000,
  "currency": "EUR",
  "salaryDuration": "annually"
}
```

### Get Job by ID
**GET** `/jobs/:id`

#### Response
Returns the complete job object with both legacy and new salary fields.

### Get All Jobs
**GET** `/jobs`

#### Query Parameters
| Parameter | Type | Description | Example |
|-----------|------|-------------|---------|
| `minSalary` | number | Filter by minimum salary | `?minSalary=50000` |
| `maxSalary` | number | Filter by maximum salary | `?maxSalary=100000` |
| `currency` | string | Filter by currency | `?currency=USD` |
| `salaryDuration` | string | Filter by duration | `?salaryDuration=annually` |

#### Response
```json
{
  "jobs": [
    {
      "id": "job-uuid-1",
      "title": "Maritime Engineer",
      "minSalary": 65000,
      "maxSalary": 85000,
      "currency": "EUR",
      "salaryDuration": "annually",
      // ... other fields
    }
  ],
  "total": 1,
  "page": 1,
  "limit": 10
}
```

## Validation Rules

### Salary Range Validation
- If both `minSalary` and `maxSalary` are provided, `minSalary` must be ≤ `maxSalary`
- Salary values must be non-negative numbers
- Maximum precision: 10 digits with 2 decimal places

### Currency Validation
- Must be one of the supported ISO 4217 currency codes
- Case-sensitive (uppercase required)

### Duration Validation
- Must be one of: `hourly`, `daily`, `weekly`, `monthly`, `annually`
- Case-sensitive (lowercase required)

### Field Dependencies
- `currency` and `salaryDuration` are recommended when salary amounts are provided
- All salary fields are optional and can be null

## Error Responses

### Validation Errors
```json
{
  "statusCode": 400,
  "message": [
    "minSalary must be less than or equal to maxSalary",
    "currency must be a valid ISO 4217 currency code"
  ],
  "error": "Bad Request"
}
```

### Supported Error Messages
- `"minSalary must be less than or equal to maxSalary"`
- `"minSalary must be a positive number"`
- `"maxSalary must be a positive number"`
- `"currency must be a valid ISO 4217 currency code"`
- `"salaryDuration must be one of: hourly, daily, weekly, monthly, annually"`

## Backward Compatibility

### Legacy Support
- The `salaryRange` field is maintained for backward compatibility
- Existing jobs retain their original `salaryRange` values
- New jobs may have `salaryRange` as null if structured salary is used

### Migration Strategy
- Existing API clients continue to work without changes
- New clients can use structured salary fields
- Both formats are returned in API responses during transition period

### Deprecation Timeline
- **Phase 1** (Current): Both formats supported
- **Phase 2** (Future): `salaryRange` marked as deprecated
- **Phase 3** (Future): `salaryRange` removed (major version update)

## Examples

### Job with Hourly Rate
```json
{
  "title": "Maritime Technician",
  "minSalary": 25.50,
  "maxSalary": 35.00,
  "currency": "USD",
  "salaryDuration": "hourly"
}
```

### Job with Monthly Salary
```json
{
  "title": "Ship Captain",
  "minSalary": 8000,
  "maxSalary": 12000,
  "currency": "EUR",
  "salaryDuration": "monthly"
}
```

### Job without Salary Information
```json
{
  "title": "Maritime Consultant",
  "minSalary": null,
  "maxSalary": null,
  "currency": null,
  "salaryDuration": null,
  "salaryRange": "Competitive salary based on experience"
}
```

## Client Implementation Guidelines

### Mobile App Integration
1. **Input Components**: Use dropdown selectors for currency and duration
2. **Validation**: Implement client-side validation for salary ranges
3. **Display**: Format salary information according to currency standards
4. **Fallback**: Handle cases where salary information is not provided

### Web Client Integration
1. **Form Handling**: Support both legacy and new salary input methods
2. **Search Filters**: Implement salary range filtering capabilities
3. **Display Formatting**: Use appropriate currency symbols and formatting
4. **Progressive Enhancement**: Enhance existing salary displays with structured data

## Testing Recommendations

### API Testing
- Test job creation with various salary combinations
- Validate error handling for invalid salary ranges
- Test backward compatibility with legacy clients
- Verify currency and duration enum validation

### Integration Testing
- Test end-to-end job posting workflow
- Validate salary display formatting
- Test search and filtering functionality
- Verify mobile app integration

---

**Version**: 1.0  
**Last Updated**: January 2024  
**Contact**: Backend Development Team